// $Id: CXMLDocument.cpp,v 1.5 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CXMLDocument.hpp"
#include "../Basics/CStringTokeniser.hpp"
#include "../Testing/CTrace.hpp"
using Exponent::Basics::CStringTokeniser;
using Exponent::IO::CXMLDocument;
using Exponent::Testing::CTrace;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CXMLDocument, CCountedObject);

//	===========================================================================
CXMLDocument::CXMLDocument() : m_rootNode(NULL), m_indentLevel(0), m_readNode(NULL)
{
	EXPONENT_CLASS_CONSTRUCTION(CXMLDocument);
}

//	===========================================================================
CXMLDocument::~CXMLDocument()
{
	EXPONENT_CLASS_DESTRUCTION(CXMLDocument);
	FORGET_COUNTED_OBJECT(m_rootNode);
}

//	===========================================================================
bool CXMLDocument::readFile(const CSystemString &filename)
{
	// Open the stream
	CTextStream stream(filename, CTextStream::e_input);

	// Check the stream is open for writing
	if (!stream.isStreamOpen())
	{
		CTrace::trace(filename, "Failed to open XML stream with path = ");
		return false;
	}

	// Read the DTD
	if (!this->readDTD(stream))
	{
		CTrace::trace(filename, "Failed to read XML DTD on path = ");
		return false;
	}

	// Check we have a root node
	FORGET_COUNTED_OBJECT(m_rootNode);

	// The string we are reading in
	CString string;

	// Loop through the document
	while(!stream.hasReachedEOF())
	{
		// REad the string in
		stream >> string;

		// Strip the space
		string.removeTrailingAndLeadingWhiteSpace();

		// Process the string
		this->processString(string, stream);
	}

	return true;
}

//	===========================================================================
bool CXMLDocument::writeFile(const CSystemString &filename)
{
	// Check we have a root node
	if (m_rootNode == NULL)
	{
		return false;
	}

	// Reset the indent
	m_indentLevel = 0;

	// Open the stream
	CTextStream stream(filename, CTextStream::e_output);

	// Check the stream is open for writing
	if (!stream.isStreamOpen())
	{
		CTrace::trace(filename, "Failed to open XML stream with path = ");
		return false;
	}

	// Now we want to output the DTD header
	this->outputDTD(stream);

	// Weite the document
	this->outputNode(m_rootNode, stream);

	// We are done!
	return true;
}

//	===========================================================================
const CXMLNode *CXMLDocument::getRootNode() const
{
	return m_rootNode;
}

//	===========================================================================
CXMLNode *CXMLDocument::getMutableRootNode()
{
	return m_rootNode;
}

//	===========================================================================
void CXMLDocument::setRootNode(CXMLNode *root)
{
	EXCHANGE_COUNTED_OBJECTS(m_rootNode, root);
}

//	===========================================================================
void CXMLDocument::outputDTD(CTextStream &stream)
{
	stream << "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n";
}

//	===========================================================================
void CXMLDocument::indentOutput(CTextStream &stream)
{
	for (long i = 0; i < m_indentLevel; i++)
	{
		stream << "\t";
	}
}

//	===========================================================================
void CXMLDocument::outputNode(CXMLNode *node, CTextStream &stream)
{
	// Now output the node
	const bool closeTag = node->getNumberOfChildNodes() == 0;

	CString name = node->getNodeName();
	this->replaceIllegalCharacters(name);

	// Output the name
	stream << "<" << name;//node->getNodeName();

	for (long i = 0; i < node->getNumberOfAttributes(); i++)
	{
		// Get the attribute
		CXMLAttribute *attribute = node->getAttribute(i);

		// Check its valid
		if (attribute == NULL)
		{
			continue;
		}

		// Copy the nae and value
		CString attributeName  = attribute->getName();
		CString attributeValue = attribute->getValue();

		// REplace the illegal characters
		this->replaceIllegalCharacters(attributeName);
		this->replaceIllegalCharacters(attributeValue);

		// Now write the attribute
		stream << " " << attributeName << "=\"" << attributeValue << "\"";
	}

	// Check if we need to close the tag in a single line
	if (closeTag)
	{
		stream << "/>\n";
	}
	else
	{
		// Close parent tag
		stream << ">\n";

		// Output children, indent
		m_indentLevel++;

		// Write the children
		for (long j = 0; j < node->getNumberOfChildNodes(); j++)
		{
			// Get the child
			CXMLNode *child = node->getChildNode(j);

			// Check its valid
			if (child == NULL)
			{
				continue;
			}

			// Indent
			this->indentOutput(stream);

			// Otherwise we have child to output
			this->outputNode(child, stream);
		}

		// Output children, indent
		m_indentLevel--;

		// Indent
		this->indentOutput(stream);

		if (node != m_rootNode)
		{
			// Output the end of the stream
			stream << "</" << name << ">\n";
		}
		else
		{
			// Output the end of the stream
			stream << "</" << name << ">";
		}

	}
}

//	===========================================================================
bool CXMLDocument::readDTD(CTextStream &stream)
{
	// The string to read
	CString dtd;

	// Read the stream
	stream >> dtd;

	// Strip the space
	dtd.removeTrailingAndLeadingWhiteSpace();

	// Find the string
	return dtd.getSubString(0, 4) == "<?xml";
}

//	===========================================================================
bool CXMLDocument::processString(CString &string, CTextStream &stream)
{
	// Check if it opens with the correct character
	if (string[0] != '<')
	{
		return false;
	}

	// Remove the leading <
	string = string.getSubString(1, string.getNumberOfCharacters() - 1);

	// Get the name
	CString name;

	// Handle based on the first character
	switch(string[0])
	{
		case '?':
		case '!':
			// Comment or declaration - we ignore
			return true;
		break;
		case '/':
			// Close case - Move back up to the parent
			if (m_readNode)
			{
				m_readNode = m_readNode->getParentNode();
			}
		break;
		default:
			// Some other kind of tag
			{
				// Find the space breaking to the attributes
				const long index =  string.findForward(' ');

				// If it doesnt exist
				if (index == -1)
				{
					const long close = string.findBackwards('>');

					if (close == -1)
					{
						// Then the string is the name
						name = string;
					}
					else
					{
						// Remove trailing '>'
						name = string.getSubString(0, close - 1);
					}

					// Remove spaces
					name.removeTrailingAndLeadingWhiteSpace();

					// Replace the escape characters
					this->replaceEscapeCharacters(name);

					// Create the node we are writing to
					if (m_rootNode == NULL)
					{
						m_rootNode = new CXMLNode(name, NULL);
						m_readNode = m_rootNode;
					}
					else
					{
						CXMLNode *node = new CXMLNode(name, m_readNode);
						m_readNode->addChildNode(node);
						m_readNode = node;
					}
				}
				else
				{
					// The name is the sub stirng
					name = string.getSubString(0, index - 1);
					name.removeTrailingAndLeadingWhiteSpace();

					// Replace the escape characters
					this->replaceEscapeCharacters(name);

					if (m_rootNode == NULL)
					{
						m_rootNode = new CXMLNode(name, NULL);
						m_readNode = m_rootNode;
					}
					else
					{
						CXMLNode *node = new CXMLNode(name, m_readNode);
						m_readNode->addChildNode(node);
						m_readNode = node;
					}

					// Remove the name from the string
					string = string.getSubString(index + 1, string.getNumberOfCharacters() - 1);

					// String now contains the set of attributes that are going to be used plus a closing tag
					// attribute1="Blah" attribute2="Blah"/
					// However, just to make life hard we have no way of knowing if the attribute values contain any spaces
					// Therefore we have the parse the the hard way

					// A string to write to
					CString readString = "";

					// Are we inside a quoted string
					bool quoteOpen = false;

					// The next attribute to add
					CXMLAttribute *attribute = NULL;

					// Loop through all the characters
					for (long i = 0; i < string.getNumberOfCharacters(); i++)
					{
						switch(string[i])
						{
							case '=':
								// Replace the string
								this->replaceEscapeCharacters(readString);

								// Construct the attribute giving it a name
								attribute  = new CXMLAttribute(readString, CString(""));

								// Reset the read string
								readString = CString::CSTRING_NULL_STRING;
							break;
							case '\"':
								// Are we inside a quote already, if so this is the end of the attribute alue
								if (quoteOpen)
								{
									// Replace the string
									this->replaceEscapeCharacters(readString);

									// Store the value
									attribute->setValue(readString);

									// Add the attribute
									m_readNode->addAttribute(attribute);

									// Dont need this any more -> notice that it is reference counted by the XMLNode its added to
									attribute  = NULL;

									// Reset the read string
									readString = "";
								}

								// Either opening or closing
								quoteOpen = !quoteOpen;
							break;
							case ' ':
								// We only append space if we are isnide a quote
								if (quoteOpen)
								{
									readString.appendString(string[i]);
								}
							break;
							default:
								// If the stirng is a close tage
								if (string[i] == '/' && i + 1 < string.getNumberOfCharacters() && string[i + 1] == '>')
								{
									// Move back up to the parent
									if (m_readNode)
									{
										m_readNode = m_readNode->getParentNode();
									}
								}
								else
								{
									// Otherwise we have a character to add to the string
									if (readString == "")
									{
										// Quick conver to string from char
										char buffer[2];
										buffer[0]  = string[i];
										buffer[1]  = '\0';

										// Stor ethe string
										readString = buffer;
									}
									else
									{
										// Append to the buffer we already have
										readString.appendString(string[i]);
									}
								}
							break;
						}
					}

					// Check the attribute was closed
					if (attribute != NULL)
					{
						CTrace::trace("Failed to close XML attribute tag");
						return false;
					}
				}
			}
		break;
	}

	// We are done :)
	return true;
}

//	===========================================================================
void CXMLDocument::replaceIllegalCharacters(CString &string)
{
	string.replaceCharWithString('&', "&amp;");
	string.replaceCharWithString('"', "&quot;");
	string.replaceCharWithString('>', "&gt;");
	string.replaceCharWithString('<', "&lt;");
	string.replaceCharWithString('\n', "&#10;");
	string.replaceCharWithString('\r', "&#13;");
}

//	===========================================================================
void CXMLDocument::replaceEscapeCharacters(CString &string)
{
	string.replaceStringWithChar('&', "&amp;");
	string.replaceStringWithChar('"', "&quot;");
	string.replaceStringWithChar('>', "&gt;");
	string.replaceStringWithChar('<', "&lt;");
	string.replaceStringWithChar('\n', "&#10;");
	string.replaceStringWithChar('\r', "&#13;");
}